﻿using System.Collections.Generic;
using System.Configuration;
using System.Data;
using System.Data.SqlClient;
using System.Linq;

public class ImagingDataExportRepository
{
    private const string CONNECT_STRING_NAME = "RegistryConnectionString";
    private const string SP_IMAGE_DATA_EXPORT = "[EFR].[usp_ImageDataExport]";
    private const string SP_IMAGE_DATA_EXPORT_REFERENCE_DATA = "[EFR].[usp_ImageDataExportReferenceData]";

    private static IReadOnlyList<string> _ignoreColumns = new List<string>
    {
        "PATIENT_ID",
        "PATIENT_ICN",
        "PATIENT_IMAGE_ID"
    };

    private static readonly IList<string> _stdImageBodyParts;
    private static readonly IList<string> _stdImageReasons;

    private enum ImageReportDataSetTable
    {
        Summary = 0,
        Details = 1,
        BodyPartsImaged = 2,
        ImagingReasons = 3
    }

    private enum ImageReportReferenceDataSetTable
    {
        BodyPartsImaged = 0,
        ImagingReasons = 1
    }

    static ImagingDataExportRepository()
    {
        _stdImageBodyParts = new List<string>();
        _stdImageReasons = new List<string>();

        GetReferenceData();
    }

    private static void GetReferenceData()
    {
        using (var ds = GetReferenceImageDataSet())
        {
            foreach (DataRow row in ds.Tables[(int)ImageReportReferenceDataSetTable.BodyPartsImaged].Rows)
                _stdImageBodyParts.Add(row.Field<string>("DESCRIPTION_TEXT"));

            foreach (DataRow row in ds.Tables[(int)ImageReportReferenceDataSetTable.ImagingReasons].Rows)
                _stdImageReasons.Add(row.Field<string>("DESCRIPTION_TEXT"));
        }
    }

    private static DataSet GetReferenceImageDataSet()
    {
        return GetDataSet(SP_IMAGE_DATA_EXPORT_REFERENCE_DATA);
    }

    internal DataTable GetImagingDataExport()
    {
        var imageDataSet = GetImageDataSet();

        return ConvertDataSetToResultTable(imageDataSet);
    }

    private DataSet GetImageDataSet()
    {
        return GetDataSet(SP_IMAGE_DATA_EXPORT);
    }

    private static DataSet GetDataSet(string storedProcedureName)
    {
        DataSet result = new DataSet();

        var connectionString = ConfigurationManager.ConnectionStrings[CONNECT_STRING_NAME].ConnectionString;

        using (SqlConnection connection = new SqlConnection(connectionString))
        using (SqlCommand getCommand = new SqlCommand(storedProcedureName, connection))
        {
            getCommand.CommandType = CommandType.StoredProcedure;

            connection.Open();

            using (var adapter = new SqlDataAdapter(getCommand))
            {
                adapter.Fill(result);
            }

            connection.Close();
        }

        return result;
    }

    private DataTable ConvertDataSetToResultTable(DataSet imageDataSet)
    {
        GetImageDetails(imageDataSet.Tables[(int)ImageReportDataSetTable.Summary], imageDataSet);

        imageDataSet.Tables[0].Columns.Remove("PATIENT_ID");

        return imageDataSet.Tables[0];
    }

    private void GetImageDetails(DataTable resultTable, DataSet imageDataSet)
    {
        DataTable imageDetailsTable = imageDataSet.Tables[(int)ImageReportDataSetTable.Details];

        foreach (DataRow resultRow in resultTable.Rows)
        {
            var currentPatientId = resultRow.Field<int>("PATIENT_ID");
            var patientImageCounter = 1;

            foreach(DataRow detailsRow in imageDetailsTable.Rows)
            {
                if (detailsRow.Field<int>("PATIENT_ID") == currentPatientId)
                {
                    var patientImageId = detailsRow.Field<int>("PATIENT_IMAGE_ID");
                    var image = $"IMG_{patientImageCounter++}";

                    foreach (DataColumn column in imageDetailsTable.Columns)
                    {
                        if (_ignoreColumns.Contains(column.ColumnName)) continue;

                        var currentColumn = $"{image}_{column.ColumnName}";

                        if (!resultTable.Columns.Contains(currentColumn))
                            resultTable.Columns.Add(currentColumn, column.DataType);

                        resultRow[currentColumn] = detailsRow[column.ColumnName];
                    }

                    GetBodyPartsImaged(resultTable, imageDataSet.Tables[(int)ImageReportDataSetTable.BodyPartsImaged], resultRow, image, patientImageId);
                    GetImagingReasons(resultTable, imageDataSet.Tables[(int)ImageReportDataSetTable.ImagingReasons], resultRow, image, patientImageId);
                }
            }
        }
    }

    private void GetBodyPartsImaged(
        DataTable resultTable,
        DataTable bodyPartsImagedTable, 
        DataRow resultRow, 
        string image, 
        int patientImageId)
    {
        foreach (var part in _stdImageBodyParts)
        {
            var columnName = $"{image}_{part}";

            if (!resultTable.Columns.Contains(columnName))
                resultTable.Columns.Add(columnName, typeof(string));

            resultRow[columnName] = "0";
        }

        foreach (DataRow row in bodyPartsImagedTable.Rows)
        {
            if (row.Field<int>("PATIENT_IMAGE_ID") == patientImageId)
            {
                var columnName = $"{image}_{row.Field<string>("BODY_PART_IMAGED")}";
                resultRow[columnName] = "1";
            }
        }
    }

    private void GetImagingReasons(
        DataTable resultTable,
        DataTable imagingReasonsTable, 
        DataRow resultRow, 
        string image, 
        int patientImageId)
    {
        foreach (var reason in _stdImageReasons)
        {
            var columnName = $"{image}_{reason}";

            if (!resultTable.Columns.Contains(columnName))
                resultTable.Columns.Add(columnName, typeof(string));

            resultRow[columnName] = "0";
        }

        foreach (DataRow row in imagingReasonsTable.Rows)
        {
            if (row.Field<int>("PATIENT_IMAGE_ID") == patientImageId)
            {
                var columnName = $"{image}_{row.Field<string>("IMAGE_REASON")}";
                resultRow[columnName] = "1";
            }
        }
    }
}